chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "injectCookies") {
    handleCookieInjection(request.provider);
    return true; // Indicates an asynchronous response
  }
  
  if (request.action === "destroyAllCookies") {
    destroyAllCookies()
      .then((result) => sendResponse({ ok: true, ...result }))
      .catch((error) => {
        console.error("Failed to destroy cookies:", error);
        sendResponse({ ok: false, error: error.message });
      });
    return true; // async response
  }
});

async function handleCookieInjection(providerString) {
  try {
    const providerData = JSON.parse(providerString);
    const { url, cookies } = providerData;

    if (!url || !Array.isArray(cookies)) {
      console.error("Invalid provider data:", providerData);
      return;
    }

    // Create the new tab first
    const newTab = await chrome.tabs.create({ url, active: true });

    // Inject cookies into the new tab's context
    for (const cookie of cookies) {
      const cookieDetails = {
        url: url,
        name: cookie.name,
        value: cookie.value,
        domain: cookie.domain,
        path: cookie.path,
        secure: cookie.secure,
        httpOnly: cookie.httpOnly,
        sameSite: cookie.sameSite,
      };

      // expirationDate is optional and should only be set if it exists
      if (cookie.expirationDate) {
        cookieDetails.expirationDate = cookie.expirationDate;
      }

      try {
        await chrome.cookies.set(cookieDetails);
      } catch (error) {
        console.error("Failed to set cookie:", cookie.name, error.message);
      }
    }

    console.log(`Successfully injected ${cookies.length} cookies for ${url}`);
  } catch (error) {
    console.error("Error handling cookie injection:", error);
  }
}

async function destroyAllCookies() {
  try {
    const cookies = await chrome.cookies.getAll({});
    let removed = 0;
    let failed = 0;

    for (const cookie of cookies) {
      const domain = cookie.domain.startsWith('.') ? cookie.domain.slice(1) : cookie.domain;
      const url = `${cookie.secure ? 'https' : 'http'}://${domain}${cookie.path || '/'}`;
      try {
        const result = await chrome.cookies.remove({ url, name: cookie.name });
        if (result) {
          removed++;
        } else {
          failed++;
        }
      } catch (e) {
        failed++;
      }
    }

    console.log(`Removed ${removed}/${cookies.length} cookies, failed: ${failed}`);
    return { total: cookies.length, removed, failed };
  } catch (error) {
    console.error('Error removing cookies:', error);
    throw error;
  }
}
