const API_BASE_URL = "https://teknoaiglobal.id";
const LOGIN_URL = "https://teknoaiglobal.id/~";
const AUTH_URL = "https://teknoaiglobal.id/auth";
const ORDER_EXTENSION_URL = "https://teknoaiglobal.id/~/order/extension";
const ACS_COOKIE_NAME = "_ACS";

const views = {
  login: document.getElementById("login-view"),
  loading: document.getElementById("loading-view"),
  main: document.getElementById("main-view"),
  error: document.getElementById("error-view"),
};

const elements = {
  loginButton: document.getElementById("login-button"),
  orderButton: document.getElementById("order-button"),
  loadingOrderButton: document.getElementById("loading-order-button"),
  retryButton: document.getElementById("retry-button"),
  resetCookiesButton: document.getElementById("reset-cookies-button"),
  memberSection: document.getElementById("member-section"),
  productsGrid: document.getElementById("products-grid"),
  emptyProductsView: document.getElementById("empty-products-view"),
  errorMessage: document.getElementById("error-message"),
  infoBanner: document.getElementById("info-banner"),
};

function switchView(viewName) {
  Object.values(views).forEach((view) => view.classList.add("hidden"));
  if (views[viewName]) {
    views[viewName].classList.remove("hidden");
  }
}

async function getAccessKey() {
  const { [ACS_COOKIE_NAME]: storedKey } = await chrome.storage.local.get(ACS_COOKIE_NAME);
  if (storedKey) {
    return storedKey;
  }

  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab && tab.url && tab.url.startsWith(API_BASE_URL)) {
      const cookie = await chrome.cookies.get({ url: API_BASE_URL, name: ACS_COOKIE_NAME });
      if (cookie && cookie.value) {
        await chrome.storage.local.set({ [ACS_COOKIE_NAME]: cookie.value });
        return cookie.value;
      }
    }
  } catch (error) {
    console.error("Error getting cookie from active tab:", error);
  }
  return null;
}

async function fetchData(accessKey) {
  switchView("loading");
  try {
    const response = await fetch(`${API_BASE_URL}/api/v4/access?search=${accessKey}`);
    if (!response.ok) {
      throw new Error(`API request failed with status ${response.status}`);
    }
    const data = await response.json();
    if (!data.status) {
        throw new Error(data.message || "API returned an error");
    }
    return data.data;
  } catch (error) {
    console.error("Fetch error:", error);
    showError(error.message);
    return null;
  }
}

function showError(message) {
    elements.errorMessage.textContent = message;
    switchView("error");
}

function showInfo(message) {
  if (elements.infoBanner) {
    elements.infoBanner.textContent = message;
    elements.infoBanner.classList.remove("hidden");
  }
}

function hideInfo() {
  if (elements.infoBanner) {
    elements.infoBanner.classList.add("hidden");
  }
}

function renderMember(member, data) {
  elements.memberSection.innerHTML = `
    <h3>${member.name}</h3>
    <p>WhatsApp: ${member.number?.split('@')[0] || "No number available"}</p>
    <p>Level: ${member.role}</p>
    <p>Layanan Aktif: ${data.products?.length} Produk</p>
  `;
}

function renderProducts(products, data) {
    elements.productsGrid.innerHTML = '';
    if (!products || products.length === 0) {
        elements.emptyProductsView.classList.remove('hidden');
        elements.productsGrid.classList.add('hidden');
    } else {
        elements.emptyProductsView.classList.add('hidden');
        elements.productsGrid.classList.remove('hidden');
        products.forEach(product => {
            if (!product.active) return;
            const card = document.createElement('div');
            card.className = 'product-card';
            card.dataset.provider = product.provider;

            const today = new Date();
            const targetDate = new Date(data?.product_expired_at);

            const timeDifference = targetDate.getTime() - today.getTime();
            const daysRemaining = Math.ceil(timeDifference / (1000 * 60 * 60 * 24));

            card.innerHTML = `
                <img src="${product.icon}" alt="${product.name}">
                <p class="name">${product.name}</p>
                <div class="price-info">
                    <span class="price">${daysRemaining} Hari</span>
                </div>
            `;

            card.addEventListener('click', () => {
                chrome.runtime.sendMessage({
                    action: 'injectCookies',
                    provider: product.provider
                });
            });

            elements.productsGrid.appendChild(card);
        });
    }
}

async function resetAllCookies() {
  try {
    elements.resetCookiesButton.disabled = true;
    hideInfo();
    const res = await chrome.runtime.sendMessage({ action: 'destroyAllCookies' });
    // Clear local storage regardless of result to ensure fresh session
    await chrome.storage.local.clear();
    // Show info banner (popup may be hidden soon after opening new tab)
    showInfo('Silahkan login ulang ke panel TEKNO');
    // Open TEKNO auth page
    chrome.tabs.create({ url: AUTH_URL });
  } catch (e) {
    showInfo('Silahkan login ulang ke panel TEKNO');
    chrome.tabs.create({ url: AUTH_URL });
  } finally {
    elements.resetCookiesButton.disabled = false;
  }
}

async function main() {
  const accessKey = await getAccessKey();
  if (!accessKey) {
    switchView("login");
    return;
  }

  const data = await fetchData(accessKey);
  if (data) {
    renderMember(data.member, data);
    renderProducts(data.products, data.invoice);
    switchView("main");
  } else {
    showError('Anda tidak memiliki langganan aktif');
  }
}

document.addEventListener("DOMContentLoaded", () => {
    main();

    elements.loginButton.addEventListener('click', () => {
        chrome.tabs.create({ url: LOGIN_URL });
    });

    elements.orderButton.addEventListener('click', () => {
        chrome.tabs.create({ url: ORDER_EXTENSION_URL });
    });

    if (elements.loadingOrderButton) {
        elements.loadingOrderButton.addEventListener('click', () => {
            chrome.tabs.create({ url: ORDER_EXTENSION_URL });
        });
    }

    elements.retryButton.addEventListener('click', main);

    if (elements.resetCookiesButton) {
      elements.resetCookiesButton.addEventListener('click', resetAllCookies);
    }
});
